/**
 *
 * \file        device.c
 *
 * \brief       Generic module supporting an abstraction of a hardware "device"
 *
 * \author      Pete McCormick
 *
 * \date        12/5/2007
 *
 * \note        The purpose of this is to provide a common interface to the
 *              various different ICs and embedded controllers to be found
 *              on our boards.  The common interface should help to make
 *              debugging, testing, etc. consistent between different devices.
 *              Most "devices" implement only a subset of all the "device"
 *              commands.  Each device can define its own "custom" commands if
 *              that would be useful.
 *
 *              This is also so that you don't have to define a bunch of
 *              different console commands.
 *
 *              Some examples of devices so far are an I2C bus, an SPI bus,
 *              a serial FLASH, an HDMI Transmitter, an HDMI receiver.
 *
 * \todo        Maybe add a list of custom commands to each device
 */

////////////////////////////////////////////////////////////////////////////////

#include "device.h"
#include "console.h"
#include "string_utils.h"
#include <string.h>
#include <stdlib.h>
#include "collection.h"

////////////////////////////////////////////////////////////////////////////////

static COLLECTION Devices;

typedef struct
{
    COLLECTION * pCollection;   // of devices
}ALLDEVS;


////////////////////////////////////////////////////////////////////////////////

const STRNUMPAIR DeviceCmds[] =
{
    {"INIT", DEVICE_INIT},
    {"PRINT", DEVICE_PRINT},
    {"TX", DEVICE_TX},
    {"RX", DEVICE_RX},
    {"DBGSET", DEVICE_SET_DEBUG},
    {"DBGGET", DEVICE_GET_DEBUG},
    {"LOAD", DEVICE_LOAD},
    {"SET", DEVICE_SET},
    {"GET", DEVICE_GET},
    {"DETECT", DEVICE_DETECT},
    {"TEST", DEVICE_SELFTEST},
    {"ERASE", DEVICE_ERASE},
    {"ENABLE", DEVICE_ENABLE},
    {"DISABLE", DEVICE_DISABLE},
    {"RESET", DEVICE_RESET},
    {"HALT", DEVICE_HALT},
    {"DUMP", DEVICE_DUMP},
    {"?", DEVICE_SYNTAX},
    {0, 0} // terminate the list
};
#define DEVICE_CMD_CNT (sizeof(DeviceCmds)/sizeof(DeviceCmds[0])-1)

////////////////////////////////////////////////////////////////////////////////

void DevicesPrepare(UINT32 maxDevices, UINT8*pData)
{
    CollectionCreate(&Devices, maxDevices, sizeof(DEVINFO), pData);
}

void DevicesInit(void)
{
    // allow user to register all their devices
    MyDevicesInit();
}

// crash if can't register device
void DeviceRegister(UINT32 deviceId, DEVFUNC pfDevEntryPt, UINT32 param)
{
    UINT32 i;
    DEVINFO * pDev = (DEVINFO *)CollectionMakeNewItem(&Devices,&i);

    if(!pDev)
    {
        return;
    }
    pDev->id = deviceId;    //todo - check if is unique?
    pDev->pfDevEntryPt = pfDevEntryPt;
    pDev->param = param;
}

INT32 DeviceProcessInput(UINT32 inst, UINT32 device, UINT32 cmd, UINT8 * pData, UINT32 *pDataBytes)
{
    // look up particular device
    DEVINFO * pDev = (DEVINFO *)CollectionGetAt(&Devices, device);
    INT32 result = 0;

    if(!pDev)
    {
        return -1;
    }
    if(!pDev->pfDevEntryPt)
    {
        return -2;
    }
    // call correct function
    result = pDev->pfDevEntryPt(pDev->param, cmd, pData, pDataBytes);

    return result;
}

// device i2c write addr subaddr data data data ....
// device vmux set addr data
// device hdmitx0
// device hdmitx0 0

INT32 DeviceCmd(UINT32 ignore, char * cmd)
{
    char * p = cmd;
    INT32 result;
    UINT32 devNum;
    UINT32 cmdNum;
    UINT8 data[81];
    UINT32 dataBytes;
    UINT32 i;
    UINT32 inst;
    DEVINFO * pDevice;

    if(*cmd == '?')                   // print help string
    {
        DmConsolePrintf("DEVICE LIST\r");
        DmConsolePrintf("\tlists all devices in the system\r");
        DmConsolePrintf("DEVICE {NAME} [INST] {CMDST} {BYTE0} {BYTE1} {BYTE2} ...\r");
        DmConsolePrintf("DEVICE {NAME} [INST] USER {CMDID} {BYTE0} {BYTE1} {BYTE2} ...\r");
        DmConsolePrintf("\tlists all devices in the system\r");
        DmConsolePrintf("\tValues are in hexadecimal\r");
        DmConsolePrintf("Device Commands:\r");
        for(i=0; i<DEVICE_CMD_CNT; i++)
        {
            DmConsolePrintf("\t%s\r", DeviceCmds[i].pStr);
        }
        return 0;
    }

    if(stricmp(cmd, "LIST") == 0)
    {
        DmConsolePrintf("Devices List:\r");
        for(pDevice = (DEVINFO *)CollectionGetFirst(&Devices,&i);
            pDevice;
            pDevice = (DEVINFO *)CollectionGetNext(&Devices,&i))
        {
            NumberGetString((STRNUMPAIR *)DeviceNames, i, (char*)data, sizeof(data));
            DmConsolePrintf("(%u) %s\r", i, data);
        }
        return 0;
    }

    // get device name
    p = strtok(cmd," ");
    if(!p)
    {
        return -1;
    }

    result = StringGetNumber((STRNUMPAIR *)DeviceNames, p, &devNum, 0);
    if(result != 0)
    {
        return -1;
    }

    // get inst
    p = strtok(NULL," ");
    if(!p)
    {
        return -1;
    }
    if((*p >= '0') && (*p <= '9'))
    {
        inst = atoi(p);
        // skip to cmd name
        p = strtok(NULL," ");
    }
    else
    {
        inst = 0;
    }

    if(!p)
    {
        return -1;
    }

    if(stricmp(p, "USER") == 0)
    {
        p = strtok(NULL," ");
        if(!p)
        {
            return 0;
        }
        cmdNum = atoi(p);
    }
    else
    {
        // get cmd name
        result = StringGetNumber((STRNUMPAIR *)DeviceCmds, p, &cmdNum, 0);
    }
    
    if( result < 0 )
        return -4;

    // now read data bytes, if any
    dataBytes = 0;
    while(1)
    {
        p = strtok(NULL, " ");
        if(!p)
        {
            break;
        }
        // convert to data
        data[dataBytes] = AsciiHexToNumber(p);
        dataBytes++;
    }
    // example
    // device <id/name> <feature> <data in ascii hex format>
    result = DeviceProcessInput(inst, devNum, cmdNum, data, &dataBytes);

    if(result < 0)
    {
        return -2;
    }

    if(result == DEVICE_OUTPUT_ASCII)
    {
        DmConsolePrintf((char *)data);
        return 0;
    }

    if(result == DEVICE_OUTPUT_RAW)
    {
        DmConsolePrintf("Device returned 0x%x bytes\r", dataBytes);
        if(dataBytes > sizeof(data))
        {
            dataBytes = sizeof(data);
        }
        for(i=0; i<dataBytes; i++)
        {
            DmConsolePrintf("%02x ", data[i]);
            if((i+1) % 16 == 0)
            {
                DmConsolePrintf("\r");
            }
        }
    }

    return 0;
}
